#!/bin/bash

echo "🔧 FTTH NMS - Docker Fix Setup Script"
echo "======================================"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Check if Docker is running
if ! docker --version > /dev/null 2>&1; then
    print_error "Docker is not installed or not running"
    exit 1
fi

if ! docker-compose --version > /dev/null 2>&1; then
    print_error "Docker Compose is not installed or not running"
    exit 1
fi

print_success "Docker and Docker Compose are available"

# Step 1: Stop existing containers
print_status "Stopping existing containers..."
docker-compose down --remove-orphans

# Step 2: Backup original files
print_status "Backing up original files..."
if [ ! -f "Dockerfile.backup" ]; then
    cp Dockerfile Dockerfile.backup
    print_success "Dockerfile backed up"
fi

if [ ! -f "docker/php.ini.backup" ]; then
    cp docker/php.ini docker/php.ini.backup
    print_success "php.ini backed up"
fi

# Step 3: Replace with fixed versions
print_status "Replacing with fixed configurations..."

if [ -f "Dockerfile-fixed" ]; then
    cp Dockerfile-fixed Dockerfile
    print_success "Dockerfile updated with fixed version"
else
    print_error "Dockerfile-fixed not found"
    exit 1
fi

if [ -f "docker/php-fixed.ini" ]; then
    cp docker/php-fixed.ini docker/php.ini
    print_success "php.ini updated with fixed version"
else
    print_error "docker/php-fixed.ini not found"
    exit 1
fi

# Step 4: Clean Docker cache
print_status "Cleaning Docker cache..."
docker system prune -f
docker volume prune -f

# Step 5: Remove old images
print_status "Removing old FTTH NMS images..."
docker rmi $(docker images ftthsnms_* -q) 2>/dev/null || print_warning "No old images to remove"

# Step 6: Rebuild containers
print_status "Rebuilding containers with fixed configuration..."
docker-compose build --no-cache

# Step 7: Start containers
print_status "Starting containers..."
docker-compose up -d

# Step 8: Wait for containers to be ready
print_status "Waiting for containers to start..."
sleep 15

# Step 9: Check container status
print_status "Checking container status..."
docker-compose ps

# Step 10: Test database connection
print_status "Testing database connection..."
sleep 5

# Check if MySQL is ready
MYSQL_READY=0
for i in {1..30}; do
    if docker-compose exec mysql mysql -u root -pftthsnms123 -e "SELECT 1;" > /dev/null 2>&1; then
        MYSQL_READY=1
        break
    fi
    print_status "Waiting for MySQL to be ready... ($i/30)"
    sleep 2
done

if [ $MYSQL_READY -eq 1 ]; then
    print_success "MySQL is ready"
    
    # Test database tables
    print_status "Checking database tables..."
    docker-compose exec mysql mysql -u ftthsnms_user -pftthsnms123 ftthnms -e "SHOW TABLES;" 2>/dev/null
    
else
    print_error "MySQL failed to start properly"
fi

# Step 11: Test web application
print_status "Testing web application..."
if curl -f http://localhost:8081/ > /dev/null 2>&1; then
    print_success "Web application is responding"
else
    print_warning "Web application may not be ready yet"
fi

# Step 12: Display access information
echo ""
print_success "Setup completed!"
echo "======================================"
echo "🌐 Application URL: http://localhost:8081/"
echo "🗄️  phpMyAdmin URL: http://localhost:8088/"
echo "📊 Dozzle (Logs): http://localhost:8080/"
echo ""
echo "📋 Default login credentials:"
echo "   Admin: admin / password"
echo "   Teknisi: teknisi1 / password"
echo ""
echo "🧪 Debug tools:"
echo "   Database debug: http://localhost:8081/debug_database_connection.php"
echo "   Form test: http://localhost:8081/test_form_submission.html"
echo ""
print_warning "If you still have issues:"
echo "1. Check logs: docker-compose logs -f"
echo "2. Check container status: docker-compose ps"
echo "3. Test database: docker-compose exec mysql mysql -u root -pftthsnms123"
echo "4. Restart containers: docker-compose restart"

# Step 13: Show logs for troubleshooting
echo ""
print_status "Recent container logs:"
echo "======================================"
docker-compose logs --tail=20

echo ""
print_success "Fix script completed!"
