<?php
/**
 * Debug version of items.php API dengan enhanced error handling
 * Gunakan file ini untuk debugging masalah penyimpanan data
 */

// Enhanced error reporting
error_reporting(E_ALL);
ini_set("display_errors", 1);
ini_set("log_errors", 1);
ini_set("error_log", "/tmp/ftth_api_debug.log");

// Prevent caching
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');
header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT');

// Start session safely
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: ' . ($_SERVER['HTTP_ORIGIN'] ?? 'http://localhost'));
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Requested-With, Authorization');
header('Access-Control-Allow-Credentials: true');
header('Access-Control-Max-Age: 3600');

// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Debug function
function debugLog($message, $data = null) {
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[{$timestamp}] {$message}";
    if ($data !== null) {
        $logMessage .= " | DATA: " . json_encode($data);
    }
    error_log($logMessage);
    
    // Also log to response for debugging
    global $debug_messages;
    if (!isset($debug_messages)) {
        $debug_messages = [];
    }
    $debug_messages[] = $logMessage;
}

try {
    debugLog("🚀 API ITEMS DEBUG - Starting request", [
        'method' => $_SERVER['REQUEST_METHOD'],
        'content_type' => $_SERVER['CONTENT_TYPE'] ?? 'not set',
        'session_id' => session_id(),
        'user_id' => $_SESSION['user_id'] ?? 'not set',
        'role' => $_SESSION['role'] ?? 'not set'
    ]);

    require_once '../config/database.php';
    require_once 'auth.php';

    // Test database connection
    $database = new Database();
    $db = $database->getConnection();
    
    if ($db === null) {
        debugLog("❌ Database connection failed");
        http_response_code(500);
        echo json_encode([
            'success' => false, 
            'message' => 'Database connection failed',
            'debug' => $debug_messages ?? []
        ]);
        exit();
    }
    
    debugLog("✅ Database connection successful");

    // Initialize response
    $response = array('success' => false);
    $method = $_SERVER['REQUEST_METHOD'];
    
    debugLog("📥 Processing {$method} request");

    // Handle method override
    if (isset($_POST['_method'])) {
        $method = strtoupper($_POST['_method']);
        debugLog("🔄 Method overridden to: {$method}");
    }

    switch ($method) {
        case 'POST':
            debugLog("📝 Handling POST request (CREATE)");
            
            // Enhanced debug for POST data
            debugLog("POST data received", [
                'post_count' => count($_POST),
                'files_count' => count($_FILES),
                'item_type' => $_POST['item_type'] ?? 'not set',
                'name' => $_POST['name'] ?? 'not set',
                'latitude' => $_POST['latitude'] ?? 'not set',
                'longitude' => $_POST['longitude'] ?? 'not set'
            ]);

            // Check admin permission
            if (!isAdmin()) {
                debugLog("❌ Admin permission check failed", [
                    'user_role' => $_SESSION['role'] ?? 'not set',
                    'is_admin_function' => function_exists('isAdmin') ? 'exists' : 'missing'
                ]);
                http_response_code(403);
                echo json_encode([
                    'success' => false, 
                    'message' => 'Admin permission required for create operations',
                    'debug' => $debug_messages ?? []
                ]);
                exit();
            }
            
            debugLog("✅ Admin permission check passed");

            // Extract and validate basic fields
            $item_type_id = $_POST['item_type'] ?? null;
            $name = $_POST['name'] ?? null;
            $latitude = $_POST['latitude'] ?? null;
            $longitude = $_POST['longitude'] ?? null;
            
            // Handle item_cable_type with proper ENUM validation
            $valid_cable_types = ['backbone', 'distribution', 'drop_core', 'feeder', 'branch'];
            $item_cable_type = $_POST['item_cable_type'] ?? 'distribution';
            if (empty($item_cable_type) || !in_array($item_cable_type, $valid_cable_types)) {
                $item_cable_type = 'distribution'; // Use default if invalid or empty
            }
            
            debugLog("🔧 Cable type validation", [
                'original_value' => $_POST['item_cable_type'] ?? 'not set',
                'validated_value' => $item_cable_type,
                'valid_options' => $valid_cable_types
            ]);
            
            debugLog("📊 Extracted basic fields", [
                'item_type_id' => $item_type_id,
                'name' => $name,
                'latitude' => $latitude,
                'longitude' => $longitude
            ]);

            // Validate required fields
            if (empty($item_type_id) || empty($name) || empty($latitude) || empty($longitude)) {
                debugLog("❌ Required field validation failed");
                $response['message'] = 'Required fields missing: item_type, name, latitude, longitude';
                $response['debug'] = $debug_messages ?? [];
                break;
            }

            // Simple INSERT query for testing
            try {
                debugLog("🔄 Attempting database INSERT");
                
                $query = "INSERT INTO ftth_items (item_type_id, name, latitude, longitude, item_cable_type, created_at, updated_at) 
                         VALUES (:item_type_id, :name, :latitude, :longitude, :item_cable_type, NOW(), NOW())";
                
                $stmt = $db->prepare($query);
                $stmt->bindParam(':item_type_id', $item_type_id);
                $stmt->bindParam(':name', $name);
                $stmt->bindParam(':latitude', $latitude);
                $stmt->bindParam(':longitude', $longitude);
                $stmt->bindParam(':item_cable_type', $item_cable_type);
                
                debugLog("🔄 Query prepared, executing...");
                
                if ($stmt->execute()) {
                    $item_id = $db->lastInsertId();
                    debugLog("✅ INSERT successful", ['new_item_id' => $item_id]);
                    
                    $response['success'] = true;
                    $response['message'] = 'Item created successfully';
                    $response['data'] = [
                        'id' => $item_id,
                        'item_type_id' => $item_type_id,
                        'name' => $name,
                        'latitude' => $latitude,
                        'longitude' => $longitude,
                        'item_cable_type' => $item_cable_type
                    ];
                    $response['debug'] = $debug_messages ?? [];
                } else {
                    debugLog("❌ INSERT failed - execute() returned false");
                    $errorInfo = $stmt->errorInfo();
                    debugLog("Database error info", $errorInfo);
                    
                    $response['message'] = 'Failed to create item - execute failed';
                    $response['sql_error'] = $errorInfo;
                    $response['debug'] = $debug_messages ?? [];
                }
                
            } catch (PDOException $e) {
                debugLog("❌ INSERT failed - PDO Exception", [
                    'error_message' => $e->getMessage(),
                    'error_code' => $e->getCode()
                ]);
                
                $response['message'] = 'Database error: ' . $e->getMessage();
                $response['debug'] = $debug_messages ?? [];
            }
            break;

        case 'GET':
            debugLog("📖 Handling GET request (READ)");
            
            // Simple GET all items
            try {
                $query = "SELECT i.*, it.name as item_type_name 
                         FROM ftth_items i 
                         LEFT JOIN item_types it ON i.item_type_id = it.id 
                         ORDER BY i.created_at DESC 
                         LIMIT 10";
                
                $stmt = $db->prepare($query);
                $stmt->execute();
                $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                debugLog("✅ GET successful", ['items_count' => count($items)]);
                
                $response['success'] = true;
                $response['data'] = $items;
                $response['debug'] = $debug_messages ?? [];
                
            } catch (PDOException $e) {
                debugLog("❌ GET failed", ['error' => $e->getMessage()]);
                $response['message'] = 'Failed to fetch items: ' . $e->getMessage();
                $response['debug'] = $debug_messages ?? [];
            }
            break;

        default:
            debugLog("❌ Unsupported method: {$method}");
            http_response_code(405);
            $response['message'] = 'Method not allowed';
            $response['debug'] = $debug_messages ?? [];
            break;
    }

} catch (Exception $e) {
    debugLog("❌ Fatal error", [
        'error_message' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
    
    http_response_code(500);
    $response = [
        'success' => false,
        'message' => 'Server error: ' . $e->getMessage(),
        'debug' => $debug_messages ?? []
    ];
}

// Send response
debugLog("📤 Sending response", ['success' => $response['success']]);
echo json_encode($response, JSON_PRETTY_PRINT);

?>
