# 🐳 TUTORIAL INSTALASI FTTHSNMS DENGAN DOCKER

## 📋 **Daftar Isi**
1. [System Requirements](#system-requirements)
2. [Prerequisites](#prerequisites)
3. [Download & Setup](#download--setup)
4. [Konfigurasi](#konfigurasi)
5. [Menjalankan System](#menjalankan-system)
6. [Verifikasi Instalasi](#verifikasi-instalasi)
7. [Login & Penggunaan](#login--penggunaan)
8. [Monitoring & Maintenance](#monitoring--maintenance)
9. [Troubleshooting](#troubleshooting)
10. [Advanced Configuration](#advanced-configuration)

---

## 🖥️ **System Requirements**

### **Minimum Requirements**
- **OS**: Linux/Windows/macOS dengan Docker support
- **RAM**: 4GB minimum, 8GB recommended
- **Storage**: 10GB free space minimum
- **CPU**: 2 cores minimum, 4 cores recommended
- **Network**: Internet connection untuk download images

### **Recommended Setup**
- **RAM**: 16GB untuk performance optimal
- **Storage**: 50GB+ untuk data growth
- **CPU**: 4+ cores untuk multiple users
- **Network**: Gigabit ethernet untuk monitoring

---

## ⚙️ **Prerequisites**

### **1. Install Docker & Docker Compose**

#### **Ubuntu/Debian:**
```bash
# Update package index
sudo apt update

# Install Docker
sudo apt install -y docker.io docker-compose

# Add user to docker group
sudo usermod -aG docker $USER

# Start Docker service
sudo systemctl start docker
sudo systemctl enable docker

# Verify installation
docker --version
docker-compose --version
```

#### **CentOS/RHEL:**
```bash
# Install Docker
sudo yum install -y docker docker-compose

# Start Docker service
sudo systemctl start docker
sudo systemctl enable docker

# Add user to docker group
sudo usermod -aG docker $USER
```

#### **Windows:**
1. Download **Docker Desktop** dari [docker.com](https://www.docker.com/products/docker-desktop)
2. Install dan restart computer
3. Verify: `docker --version` di Command Prompt

#### **macOS:**
1. Download **Docker Desktop** untuk Mac
2. Install dengan drag & drop ke Applications
3. Launch Docker Desktop dan tunggu sampai running

### **2. Verify Docker Installation**
```bash
# Test Docker
docker run hello-world

# Test Docker Compose
docker-compose --version

# Check Docker status
docker ps
```

**Expected Output:**
```
CONTAINER ID   IMAGE         COMMAND                  CREATED         STATUS         PORTS     NAMES
(empty list - normal untuk fresh install)
```

---

## 📁 **Download & Setup**

### **Method 1: Git Clone (Recommended)**
```bash
# Clone repository
git clone https://github.com/saputrabudi/ftthsnms.git
cd ftthsnms

# Verify files
ls -la
```

### **Method 2: Download ZIP**
```bash
# Download dan extract
wget https://github.com/saputrabudi/ftthsnms/archive/main.zip
unzip main.zip
cd ftthsnms-main

# Atau gunakan curl
curl -L https://github.com/saputrabudi/ftthsnms/archive/main.zip -o ftthsnms.zip
unzip ftthsnms.zip
cd ftthsnms-main
```

### **Method 3: Manual Setup**
Jika tidak ada Git/wget:
1. Download ZIP dari GitHub manually
2. Extract ke folder `ftthsnms`
3. Masuk ke folder tersebut

---

## 🔧 **Konfigurasi**

### **1. File Structure Verification**
Pastikan struktur file seperti ini:
```
ftthsnms/
├── docker-compose.yml     ✅ Docker orchestration
├── Dockerfile            ✅ PHP application container
├── database.sql          ✅ Database schema (cleaned)
├── fix_docker_setup.sh   ✅ Automated setup script
├── index.php             ✅ Main application
├── config/               ✅ Configuration files
├── api/                  ✅ API endpoints
├── assets/               ✅ CSS, JS, images
└── docker/               ✅ Docker configurations
```

### **2. Environment Configuration**
```bash
# Copy environment template (jika ada)
cp .env.example .env

# Edit konfigurasi (optional)
nano .env
```

### **3. Docker Compose Configuration**
File `docker-compose.yml` sudah dikonfigurasi dengan:
```yaml
services:
  mysql:
    image: mysql:8.0
    environment:
      MYSQL_ROOT_PASSWORD: ftthsnms123
      MYSQL_DATABASE: ftthnms
      MYSQL_USER: ftthsnms_user
      MYSQL_PASSWORD: ftthsnms123
    ports:
      - "3306:3306"

  phpmyadmin:
    image: phpmyadmin/phpmyadmin:latest
    environment:
      PMA_HOST: mysql
      PMA_USER: root
      PMA_PASSWORD: ftthsnms123
    ports:
      - "8088:80"

  ftthsnms_app:
    build: .
    ports:
      - "8081:80"
    depends_on:
      - mysql
```

---

## 🚀 **Menjalankan System**

### **Method 1: Automated Setup (Recommended)**
```bash
# Berikan permission ke script
chmod +x fix_docker_setup.sh

# Jalankan automated setup
./fix_docker_setup.sh
```

**Script ini akan:**
- ✅ Stop existing containers
- ✅ Remove old images 
- ✅ Build fresh containers
- ✅ Start all services
- ✅ Verify database connection
- ✅ Test application accessibility

### **Method 2: Manual Setup**
```bash
# Stop existing containers (jika ada)
docker-compose down

# Build containers
docker-compose build --no-cache

# Start services
docker-compose up -d

# Monitor logs
docker-compose logs -f
```

### **3. Verify Services**
```bash
# Check running containers
docker-compose ps

# Expected output:
NAME                   COMMAND                  SERVICE       STATUS        PORTS
ftthsnms_app           "docker-php-entrypoi…"   ftthsnms_app  running       0.0.0.0:8081->80/tcp
ftthsnms_mysql         "docker-entrypoint.s…"   mysql         running       0.0.0.0:3306->3306/tcp, 33060/tcp
ftthsnms_phpmyadmin    "/docker-entrypoint.…"   phpmyadmin    running       0.0.0.0:8088->80/tcp
```

---

## ✅ **Verifikasi Instalasi**

### **1. Check Application**
```bash
# Test web application
curl -I http://localhost:8081/

# Expected: HTTP/1.1 200 OK
```

### **2. Check Database**
```bash
# Test database connection
docker-compose exec mysql mysql -u ftthsnms_user -pftthsnms123 -e "SHOW DATABASES;"

# Expected output:
+--------------------+
| Database           |
+--------------------+
| ftthnms            |
| information_schema |
| performance_schema |
+--------------------+
```

### **3. Check Database Tables**
```bash
# Check tables
docker-compose exec mysql mysql -u ftthsnms_user -pftthsnms123 ftthnms -e "SHOW TABLES;"

# Expected: List of 15+ tables
```

### **4. Test Ping Monitoring**
```bash
# Test ping API
curl "http://localhost:8081/api/ping_monitor.php?action=ping_single&host=8.8.8.8"

# Expected: JSON response with ping status
```

---

## 🔐 **Login & Penggunaan**

### **Default Access URLs**
- **🌐 Main Application**: http://localhost:8081/
- **📊 phpMyAdmin**: http://localhost:8088/
- **🔍 Dozzle (Logs)**: http://localhost:8080/ (optional)

### **Default Login Credentials**
```
👤 Administrator:
Username: admin
Password: password
Role: Full access to all features

👤 Teknisi:
Username: teknisi  
Password: password
Role: Limited access for field technicians
```

### **First Login Steps**
1. **Open browser** dan akses http://localhost:8081/
2. **Login** dengan kredensial admin
3. **Verify dashboard** loads correctly
4. **Test basic features**:
   - ✅ Add new item to map
   - ✅ View item list
   - ✅ Check monitoring status
   - ✅ Access user management (admin only)

---

## 📊 **Monitoring & Maintenance**

### **1. Container Health Monitoring**
```bash
# Check container status
docker-compose ps

# View logs
docker-compose logs ftthsnms_app
docker-compose logs mysql
docker-compose logs phpmyadmin

# Monitor resource usage
docker stats
```

### **2. Database Maintenance**
```bash
# Backup database
docker-compose exec mysql mysqldump -u ftthsnms_user -pftthsnms123 ftthnms > backup_$(date +%Y%m%d).sql

# Restore database
docker-compose exec -T mysql mysql -u ftthsnms_user -pftthsnms123 ftthnms < backup_file.sql
```

### **3. Application Maintenance**
```bash
# Update application code
git pull origin main

# Rebuild containers
docker-compose build --no-cache ftthsnms_app
docker-compose restart ftthsnms_app

# Clear application cache (if any)
docker-compose exec ftthsnms_app rm -rf /tmp/cache/*
```

### **4. Auto Monitoring Setup**
```bash
# Setup cron untuk monitoring otomatis
crontab -e

# Add this line untuk monitoring setiap 5 menit:
*/5 * * * * cd /path/to/ftthsnms && docker-compose exec -T ftthsnms_app php auto_monitor.php >> /var/log/ftth_monitor.log 2>&1
```

---

## 🔧 **Troubleshooting**

### **Common Issues & Solutions**

#### **1. Container Won't Start**
```bash
# Check Docker daemon
sudo systemctl status docker

# Check logs
docker-compose logs

# Rebuild containers
docker-compose down
docker-compose build --no-cache
docker-compose up -d
```

#### **2. Database Connection Error**
```bash
# Check MySQL container
docker-compose logs mysql

# Reset database
docker-compose down
docker volume rm ftthsnms_mysql_data
docker-compose up -d
```

#### **3. Port Already in Use**
```bash
# Check what's using ports
sudo netstat -tulpn | grep :8081
sudo netstat -tulpn | grep :3306
sudo netstat -tulpn | grep :8088

# Kill process or change ports in docker-compose.yml
```

#### **4. Permission Issues**
```bash
# Fix file permissions
sudo chown -R $USER:$USER ./ftthsnms/
chmod -R 755 ./ftthsnms/

# Fix Docker permissions
sudo chmod 666 /var/run/docker.sock
```

#### **5. Out of Space Error**
```bash
# Clean Docker
docker system prune -f
docker volume prune -f
docker image prune -f

# Check disk space
df -h
```

#### **6. Slow Performance**
```bash
# Increase Docker memory limit
# Edit Docker Desktop settings atau:
echo '{"default-ulimits":{"nofile":{"Hard":64000,"Name":"nofile","Soft":64000}}}' | sudo tee /etc/docker/daemon.json
sudo systemctl restart docker
```

#### **7. Network Issues**
```bash
# Reset Docker network
docker network prune -f
docker-compose down
docker-compose up -d

# Check network connectivity
docker-compose exec ftthsnms_app ping mysql
docker-compose exec ftthsnms_app ping google.com
```

### **Debug Commands**
```bash
# Enter container shell
docker-compose exec ftthsnms_app bash
docker-compose exec mysql bash

# Check PHP configuration
docker-compose exec ftthsnms_app php -i | grep -E "(version|extension)"

# Test database connection dari app
docker-compose exec ftthsnms_app php debug_database_connection.php

# Monitor real-time logs
docker-compose logs -f --tail=50
```

---

## 🔰 **Advanced Configuration**

### **1. SSL/HTTPS Setup**
```bash
# Generate SSL certificate
openssl req -x509 -nodes -days 365 -newkey rsa:2048 \
  -keyout ssl/private.key -out ssl/certificate.crt

# Update docker-compose.yml
# Add SSL volume mapping dan configure Apache
```

### **2. External Database**
Edit `docker-compose.yml` untuk menggunakan external database:
```yaml
services:
  ftthsnms_app:
    environment:
      - DB_HOST=your-external-db.com
      - DB_NAME=ftthnms
      - DB_USER=your_user
      - DB_PASSWORD=your_password
```

### **3. Load Balancer Setup**
```yaml
# Add nginx load balancer
nginx:
  image: nginx:alpine
  ports:
    - "80:80"
    - "443:443"
  volumes:
    - ./nginx.conf:/etc/nginx/nginx.conf
  depends_on:
    - ftthsnms_app
```

### **4. Production Environment**
```bash
# Production docker-compose
cp docker-compose.yml docker-compose.prod.yml

# Edit untuk production settings:
# - Remove phpMyAdmin
# - Add SSL termination
# - Configure backup volumes
# - Set restart policies
# - Configure logging
```

### **5. Backup & Recovery Strategy**
```bash
# Create backup script
cat > backup.sh << 'EOF'
#!/bin/bash
DATE=$(date +%Y%m%d_%H%M%S)
docker-compose exec -T mysql mysqldump -u ftthsnms_user -pftthsnms123 ftthnms > backup_${DATE}.sql
tar -czf backup_${DATE}.tar.gz backup_${DATE}.sql docker-compose.yml .env
echo "Backup completed: backup_${DATE}.tar.gz"
EOF

chmod +x backup.sh
```

---

## 🎯 **Performance Optimization**

### **1. MySQL Optimization**
Edit `docker-compose.yml`:
```yaml
mysql:
  command: --default-authentication-plugin=mysql_native_password 
           --innodb-buffer-pool-size=1G 
           --max-connections=200
           --query-cache-size=64M
```

### **2. PHP Optimization**
Update `docker/php.ini`:
```ini
memory_limit = 512M
max_execution_time = 300
max_input_vars = 3000
opcache.enable = 1
opcache.memory_consumption = 128
```

### **3. Apache Optimization**
```apache
# Add to .htaccess
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript
</IfModule>

<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 month"
    ExpiresByType image/jpeg "access plus 1 month"
    ExpiresByType image/gif "access plus 1 month"
    ExpiresByType image/png "access plus 1 month"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType application/pdf "access plus 1 month"
    ExpiresByType text/javascript "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
</IfModule>
```

---

## 📞 **Support & Documentation**

### **Useful Resources**
- 📖 **Main Documentation**: README.md
- 🔧 **Database Issues**: DATABASE_ANALYSIS.md
- 🔍 **Ping Monitoring**: PING_MONITORING_SOLUTION.md
- 🐛 **Form Issues**: FIX_CABLE_TYPE_ENUM_ISSUE.md

### **Getting Help**
1. **Check logs**: `docker-compose logs`
2. **Check documentation** files in project
3. **Test with debug tools** provided
4. **Contact support** dengan error details

### **Quick Commands Reference**
```bash
# Start system
docker-compose up -d

# Stop system
docker-compose down

# View logs
docker-compose logs -f

# Backup database
docker-compose exec mysql mysqldump -u ftthsnms_user -pftthsnms123 ftthnms > backup.sql

# Update application
git pull && docker-compose build --no-cache ftthsnms_app && docker-compose restart ftthsnms_app

# Emergency reset
docker-compose down && docker system prune -f && docker-compose up -d
```

---

## ✅ **Installation Checklist**

### **Pre-Installation**
- [ ] Docker & Docker Compose installed
- [ ] Sufficient disk space (10GB+)
- [ ] Network connectivity verified
- [ ] Ports 8081, 3306, 8088 available

### **Installation Process**
- [ ] Project files downloaded/cloned
- [ ] File permissions configured
- [ ] Docker containers built successfully
- [ ] All services started (3 containers running)
- [ ] Database tables created automatically

### **Post-Installation Verification**
- [ ] Web interface accessible (http://localhost:8081)
- [ ] Login works with default credentials
- [ ] Database connection established
- [ ] phpMyAdmin accessible (http://localhost:8088)
- [ ] Ping monitoring functional
- [ ] Basic CRUD operations work

### **Production Readiness**
- [ ] SSL certificate configured (for production)
- [ ] Backup strategy implemented
- [ ] Monitoring setup configured
- [ ] Default passwords changed
- [ ] Firewall rules configured
- [ ] Resource limits set appropriately

---

## 🎉 **Selamat!**

Instalasi FTTHSNMS berhasil! Anda sekarang memiliki sistem manajemen jaringan FTTH yang lengkap dengan:

✅ **Web-based interface** untuk management  
✅ **Real-time monitoring** dengan ping status  
✅ **Database management** via phpMyAdmin  
✅ **User authentication** dengan role-based access  
✅ **Network mapping** dengan coordinates  
✅ **Cable routing** planning tools  
✅ **Item management** untuk semua equipment FTTH  
✅ **Automated monitoring** system  

**Next Steps:**
1. Login dan explore interface
2. Add your first FTTH items
3. Configure monitoring untuk devices
4. Setup automated backups
5. Train users pada system usage

**Happy monitoring!** 🚀

---

*Tutorial ini dibuat untuk FTTH Schematic Network Management System oleh Saputra Budi*
